<?php
/**
 * Format Number
 *
 * @package     AutomatorWP\Integrations\Formatter\Actions\Format_Number
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Formatter_Format_Number extends AutomatorWP_Integration_Action {

    public $integration = 'formatter';
    public $action = 'formatter_format_number';
    public $formatted = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Convert number into format', 'automatorwp-pro' ),
            'select_option'     => __( 'Convert <strong>number</strong> into <strong>format</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Number. %2$s: Format. */
            'edit_label'        => sprintf( __( 'Convert %1$s into %2$s', 'automatorwp-pro' ), '{number}', '{format}' ),
            /* translators: %1$s: Number. %2$s: Format. */
            'log_label'         => sprintf( __( 'Convert %1$s into %2$s', 'automatorwp-pro' ), '{number}', '{format}' ),
            'options'           => array(
                'number' => array(
                    'from'  => 'number',
                    'default' => __( 'number', 'automatorwp-pro' ),
                    'fields' => array(
                        'number' => array(
                            'name' => __( 'Number:', 'automatorwp-pro' ),
                            'desc' => __( 'Number to convert into format.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                            'default' => ''
                        ),
                     ) ),
                'format' => array(
                    'from'  => 'format',
                    'default' => __( 'format', 'automatorwp-pro' ),
                    'fields' => array(
                        'format' => array(
                            'name' => __( 'Format:', 'automatorwp-generator' ),
                            'desc' => __( 'Format to apply to the number.', 'automatorwp-pro' )
                                . ' ' . '<a href="https://automatorwp.com/docs/formatter/number-format/" target="_blank">' . __( 'More info', 'automatorwp-pro' ) . '</a>',
                            'type' => 'text',
                            'required' => true,
                            'default' => '#,###.00'
                        ),
                    ) ),
            ),
            'tags'  => automatorwp_formatter_get_actions_string_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $number = (float)$action_options['number'];
        $format = $action_options['format'];

        // Bail if not number
        if ( empty( $number ) ) {
            $this->result = __( 'Please, insert a number to be formatted', 'automatorwp-pro' );
            return;
        }

        // Bail if not position
        if ( empty( $format ) ) {
            $this->result = __( 'Please, insert a format', 'automatorwp-pro' );
            return;
        }

        $thousands_separator = '';
        $decimal_separator = '';
        $decimals = 0;

        $thousands_open = false;
        $decimals_open = false;
        $thousands_str = '';
        $decimals_str = '';

        for ($i = 0; $i < strlen($format); $i++) {

            // Thousands
            if( $format[$i] === '#' ) {

                $thousands_open = true;
                $decimals_open = false;

                // Thousands separator
                if( isset( $format[$i-1] ) && $format[$i-1] !== '#' && $format[$i-1] !== ' ' ) {
                    $thousands_separator = $format[$i-1];
                }
            }

            // Decimals
            if( $format[$i] === '0' ) {

                $thousands_open = false;
                $decimals_open = true;

                // Decimals separator
                if( isset( $format[$i-1] ) && $format[$i-1] !== '0' && $format[$i-1] !== ' ' ) {
                    $decimal_separator = $format[$i-1];
                }

                // Increment the number of decimals for the number_format()
                $decimals++;
            }

            // Build a string patter to detect the thousands side
            if( $thousands_open ) {
                $thousands_str .= $format[$i];
            }

            // Build a string patter to detect the decimals side
            if( $decimals_open && $format[$i] === '0' ) {
                $decimals_str .= $format[$i];
            }
        }

        // Format the number
        $number_format = number_format( $number, $decimals, $decimal_separator, $thousands_separator );
        $final_number = $format;

        if( $decimals > 0 ) {
            // Remove the decimal separator
            $thousands_str = str_replace( $decimal_separator, '', $thousands_str );
            // Explode the formatted number to replace the thousands and decimals sides
            $number_format = explode( $decimal_separator, $number_format );

            // Replace the thousands and decimals
            $final_number = str_replace( $thousands_str, $number_format[0], $final_number );
            $final_number = str_replace( $decimals_str, $number_format[1], $final_number );
        } else {
            // Replace the thousands only
            $final_number = str_replace( $thousands_str, $number_format, $final_number );
        }

        $this->formatted = $final_number;

        $this->result = sprintf( __( '%s', 'automatorwp-pro' ), $this->formatted );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['formatted'] = ( isset( $this->formatted ) ? $this->formatted : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Formatter_Format_Number();